<?php

namespace App\Filament\Resources\BtsEnergyAnalysisResource\Pages;

use App\Filament\Resources\BtsEnergyAnalysisResource;
use App\Models\BtsEnergyAnalysis;
use App\Jobs\ProcessBtsEnergyAnalysisJob;
use App\Jobs\ProcessNewBtsWithReferenceJob;
use Filament\Actions;
use Filament\Resources\Pages\ListRecords;
use Filament\Forms;
use Filament\Notifications\Notification;
use Illuminate\Support\Facades\Response;
use Illuminate\Support\Facades\Auth;
use PhpOffice\PhpSpreadsheet\IOFactory;
use Illuminate\Support\Facades\Storage;

class ListBtsEnergyAnalyses extends ListRecords
{
    protected static string $resource = BtsEnergyAnalysisResource::class;

    protected function getHeaderActions(): array
    {
        return [
            Actions\Action::make('upload_file')
                ->label('Upload File')
                ->icon('heroicon-o-cloud-arrow-up')
                ->form([
                    Forms\Components\Select::make('file_type')
                        ->label('Loại dữ liệu')
                        ->options([
                            'monthly_detail' => 'Có Month_id (dữ liệu chi tiết theo tháng)',
                            'monthly_avg' => 'Không có Month_id (dữ liệu trung bình)'
                        ])
                        ->required(),
                    Forms\Components\Hidden::make('analysis_role')
                        ->default('reference'),
                    Forms\Components\FileUpload::make('data_file')
                        ->label('File Excel')
                        ->acceptedFileTypes(['application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'])
                        ->required()
                        ->disk('local')
                        ->directory('bts-energy-uploads')
                        ->preserveFilenames()
                        ->getUploadedFileNameForStorageUsing(function ($file) {
                            $name = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
                            $ext  = $file->getClientOriginalExtension();
                            $now = now()->format('Y_m_d_H_i_s');
                            return $name . '_' . $now . '.' . $ext;
                        })
                ])
                ->action(function (array $data) {
                    return $this->uploadFile($data);
                }),
            Actions\Action::make('upload_new_bts_with_reference')
                ->label('Upload file đánh giá trạm theo file trước đó')
                ->icon('heroicon-o-arrow-up-on-square-stack')
                ->form([
                    Forms\Components\Select::make('reference_analysis_id')
                        ->label('Chọn file đã xử lý trước đó')
                        ->options(function () {
                            return BtsEnergyAnalysis::query()
                                ->where('status', 'completed')
                                ->where('analysis_role', 'reference')
                                ->orderByDesc('id')
                                ->limit(200)
                                ->get()
                                ->pluck('file_name', 'id');
                        })
                        ->searchable()
                        ->required(),
                    Forms\Components\FileUpload::make('data_file')
                        ->label('File BTS mới (Excel)')
                        ->acceptedFileTypes(['application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'])
                        ->required()
                        ->disk('local')
                        ->directory('bts-energy-uploads')
                        ->preserveFilenames()
                        ->getUploadedFileNameForStorageUsing(function ($file) {
                            $name = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
                            $ext  = $file->getClientOriginalExtension();
                            $now = now()->format('Y_m_d_H_i_s');
                            return $name . '_' . $now . '.' . $ext;
                        }),

                    Forms\Components\Hidden::make('file_type'),

                ])
                ->action(function (array $data) {
                    $referenceAnalysis = BtsEnergyAnalysis::find($data['reference_analysis_id']);
                    $data['file_type'] = $referenceAnalysis->file_type;
                    
                    $uploadedFilePath = Storage::disk('local')->path($data['data_file']);
                    $fileType = $this->detectFileType($uploadedFilePath);

                    if ($fileType !== $data['file_type']) {
                        Notification::make()
                            ->title('Sai loại file')
                            ->body("File tham chiếu là '{$data['file_type']}', nhưng bạn upload file '{$fileType}' khác loại.")
                            ->danger()
                            ->send();
                        return;
                    }
                    return $this->uploadNewBtsWithReference($data);
                }),
            Actions\Action::make('download_template')
                ->label('Download Template')
                ->icon('heroicon-o-document-arrow-down')
                ->form([
                    Forms\Components\Select::make('template_type')
                        ->label('Loại dữ liệu')
                        ->options([
                            'with_month_id' => 'Có Month_id (dữ liệu chi tiết theo tháng)',
                            'without_month_id' => 'Không có Month_id (dữ liệu trung bình)'
                        ])
                        ->required()
                ])
                ->action(function (array $data) {
                    return $this->downloadTemplate($data['template_type']);
                })
        ];
    }
    private function detectFileType(string $filePath): string
    {
        $spreadsheet = IOFactory::load($filePath);
        $sheet = $spreadsheet->getActiveSheet();
        $headerRow = $sheet->rangeToArray('A1:' . $sheet->getHighestColumn() . '1')[0];
        $headerRow = array_values(array_filter($headerRow, fn($value) => $value !== null && trim($value) !== ''));

        $monthlyDetailHeaders = [
            'STT',
            'Month_id',
            'BTS_id',
            'Traffic_2G',
            'Data_2G',
            'AMR_TRAF',
            'DATA3G',
            'TRAFFIC_4G',
            'P_SubscribedUser',
            'E_PowerConsumption_AC',
            'E_PowerConsumption_TBVT',
            'E_PowerConsumption_TBTD',
            'E_PowerConsumption_MAYLANH',
            'Distinct_Attribute_PIN_NLMT'
        ];

        $monthlyAvgHeaders = [
            'STT',
            'BTS_id',
            'P_SubscribedUser',
            'P_VoiceTraffic',
            'P_DataTraffic',
            'E_PowerConsumption_AC',
            'E_PowerConsumption_TBVT',
            'E_PowerConsumption_TBTD',
            'E_PowerConsumption_MAYLANH',
            'Distinct_Attribute_PIN_NLMT'
        ];

        if ($headerRow === $monthlyDetailHeaders) {
            return 'monthly_detail';
        } elseif ($headerRow === $monthlyAvgHeaders) {
            return 'monthly_avg';
        } else {
            return 'unknown';
        }
    }
    private function downloadTemplate(string $templateType)
    {
        try {
            $columns = $this->getTemplateColumns($templateType);
            
            // Tạo file Excel
            $filename = 'bts_energy_template_' . $templateType . '_' . date('Y_m_d_H_i_s') . '.xlsx';
            
            // Tạo nội dung file
            $content = $this->generateExcelContent($columns);
            
            // Trả về file download
            return Response::streamDownload(function () use ($content) {
                echo $content;
            }, $filename, [
                'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
                'Content-Disposition' => 'attachment; filename="' . $filename . '"'
            ]);
        } catch (\Exception $e) {
            Notification::make()
                ->title('Lỗi tạo file')
                ->body('Không thể tạo file template: ' . $e->getMessage())
                ->danger()
                ->send();
        }
    }

    private function getTemplateColumns(string $templateType): array
    {
        switch ($templateType) {
            case 'with_month_id':
                return [
                    'STT',
                    'Month_id',
                    'BTS_id',
                    'Traffic_2G',
                    'Data_2G',
                    'AMR_TRAF',
                    'DATA3G',
                    'TRAFFIC_4G',
                    'P_SubscribedUser',
                    'E_PowerConsumption_AC',
                    'E_PowerConsumption_TBVT',
                    'E_PowerConsumption_TBTD',
                    'E_PowerConsumption_MAYLANH',
                    'Distinct_Attribute_PIN_NLMT'
                ];
                
            case 'without_month_id':
                return [
                    'STT',
                    'BTS_id',
                    'P_SubscribedUser',
                    'P_VoiceTraffic',
                    'P_DataTraffic',
                    'E_PowerConsumption_AC',
                    'E_PowerConsumption_TBVT',
                    'E_PowerConsumption_TBTD',
                    'E_PowerConsumption_MAYLANH',
                    'Distinct_Attribute_PIN_NLMT'
                ];
                
            default:
                return [];
        }
    }

    private function generateExcelContent(array $columns): string
    {
        // Tạo file tạm để tránh lỗi memory
        $tempFile = tempnam(sys_get_temp_dir(), 'bts_energy_template');
        
        try {
            $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
            $worksheet = $spreadsheet->getActiveSheet();
            
            // Thêm headers
            $col = 'A';
            foreach ($columns as $header) {
                $worksheet->setCellValue($col . '1', $header);
                $col++;
            }
            
            // Tự động giãn column theo nội dung
            foreach (range('A', $col) as $columnID) {
                $worksheet->getColumnDimension($columnID)->setAutoSize(true);
            }
            
            // Lưu vào file tạm
            $writer = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);
            $writer->setPreCalculateFormulas(false);
            $writer->save($tempFile);
            
            // Đọc nội dung file
            $content = file_get_contents($tempFile);
            
            return $content;
        } finally {
            // Xóa file tạm
            if (file_exists($tempFile)) {
                unlink($tempFile);
            }
        }
    }
    
    public function downloadResult($resultFile)
    {
        $filePath = storage_path('app/bts-analysis/results/' . $resultFile);
        
        if (file_exists($filePath)) {
            return Response::download($filePath);
        }
        
        abort(404, 'File không tồn tại');
    }

    private function uploadFile(array $data)
    {
        try {
            // Lấy tên file từ đường dẫn
            $fileName = basename($data['data_file']);
            
            // Tạo record trong database
            $analysis = BtsEnergyAnalysis::create([
                'file_name' => $fileName,
                'file_type' => $data['file_type'],
                'analysis_role' => $data['analysis_role'] ?? 'standalone',
                'status' => 'pending',
                'description' => 'File upload để phân tích năng lượng BTS',
                'uploaded_by' => Auth::id(),
            ]);

            // Dispatch job để xử lý file
            ProcessBtsEnergyAnalysisJob::dispatch($analysis->id, $data['data_file'], $data['file_type'], Auth::id());

            Notification::make()
                ->title('Upload thành công')
                ->body('File đã được upload và đang được xử lý. Vui lòng chờ...')
                ->success()
                ->send();

        } catch (\Exception $e) {
            Notification::make()
                ->title('Lỗi upload file')
                ->body('Không thể upload file: ' . $e->getMessage())
                ->danger()
                ->send();
        }
    }

    private function uploadNewBtsWithReference(array $data)
    {
        try {
            $fileName = basename($data['data_file']);

            $analysis = BtsEnergyAnalysis::create([
                'file_name' => $fileName,
                'file_type' => $data['file_type'],
                'analysis_role' => 'derived',
                'reference_analysis_id' => (int) $data['reference_analysis_id'],
                'status' => 'pending',
                'description' => 'Upload BTS mới dựa trên file tham chiếu #' . $data['reference_analysis_id'],
                'uploaded_by' => Auth::id(),
            ]);

            ProcessNewBtsWithReferenceJob::dispatch(
                $analysis->id,
                $data['data_file'],
                $data['file_type'],
                (int) $data['reference_analysis_id'],
                Auth::id()
            );

            Notification::make()
                ->title('Upload thành công')
                ->body('File mới đã được upload và đang xử lý theo file tham chiếu. Vui lòng chờ...')
                ->success()
                ->send();

        } catch (\Exception $e) {
            Notification::make()
                ->title('Lỗi upload file')
                ->body('Không thể upload/khởi chạy xử lý: ' . $e->getMessage())
                ->danger()
                ->send();
        }
    }
}
