<?php

namespace App\Filament\Resources\BtsRankingResource\Pages;

use App\Filament\Resources\BtsRankingResource;
use App\Jobs\ProcessBtsRankingJob;
use App\Models\BtsRanking;
use Filament\Actions;
use Filament\Resources\Pages\ListRecords;
use Filament\Forms;
use Filament\Notifications\Notification;
use Illuminate\Support\Facades\Response;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Auth;

class ListBtsRankings extends ListRecords
{
    protected static string $resource = BtsRankingResource::class;

    protected function getHeaderActions(): array
    {
        return [
            // Actions\CreateAction::make(),
            Actions\Action::make('upload_file')
                ->label('Upload File')
                ->icon('heroicon-o-arrow-up-tray')
                ->form([
                    Forms\Components\FileUpload::make('file')
                        ->label('Chọn file Excel')
                        ->acceptedFileTypes(['application/vnd.openxmlformats-officedocument.spreadsheetml.sheet', 'text/csv'])
                        ->required()
                        ->directory('bts-ranking')
                        ->disk('local')
                        ->preserveFilenames()
                        ->getUploadedFileNameForStorageUsing(function ($file) {
                            $name = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
                            $ext  = $file->getClientOriginalExtension();
                            $now = now()->format('Y_m_d_H_i_s');
                            return $name . '_' . $now . '.' . $ext;
                        })
                ])
                ->action(function (array $data) {
                    return $this->uploadFile($data);
                }),
            Actions\Action::make('download_template')
                ->label('Download Template')
                ->icon('heroicon-o-document-arrow-down')
                ->form([
                    Forms\Components\Select::make('template_type')
                        ->label('Loại template')
                        ->options([
                            'ranking' => 'Template BTS ranking MPĐ'
                        ])
                        ->required()
                        ->default('ranking')
                ])
                ->action(function (array $data) {
                    return $this->downloadTemplate($data['template_type']);
                })
        ];
    }
    private function downloadTemplate(string $templateType)
    {
        try {
            $columns = $this->getTemplateColumns($templateType);
            
            // Tạo file Excel
            $filename = 'bts_ranking_template_' . $templateType . '_' . date('Y_m_d_H_i_s') . '.xlsx';
            
            // Tạo nội dung file
            $content = $this->generateExcelContent($columns);
            
            // Trả về file download
            return Response::streamDownload(function () use ($content) {
                echo $content;
            }, $filename, [
                'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
                'Content-Disposition' => 'attachment; filename="' . $filename . '"'
            ]);
        } catch (\Exception $e) {
            Notification::make()
                ->title('Lỗi tạo file')
                ->body('Không thể tạo file template: ' . $e->getMessage())
                ->danger()
                ->send();
        }
    }
    private function getTemplateColumns(string $templateType): array
    {
        switch ($templateType) {
            case 'ranking':
                return [
                    'STT',
                    'SITE_ID',
                    'OBJ_ID',
                    'MPD_ID',
                    'NHAN_HIEU_MPD',
                    'CONG_SUAT_MPD',
                    'LUONG_NHIEN_LIEU_SU_DUNG (lít/ngày)',
                    'CONG_SUAT_DINH_MUC',
                    'CONG_SUAT_TOI_DA',
                    'CONSUMPTION_RATE_FULLLOAD',
                    'LOAI_MAY_PHAT_DIEN',
                    'NHIEN_LIEU',
                    'THOI_GIAN_SU_DUNG (phút)'
                ];
                
            default:
                return [];
        }
    }
    private function uploadFile(array $data)
    {
        try {
            $filePath = $data['file'];
            $description = $data['description'] ?? '';
            
            // Lấy thông tin file từ path
            $fileName = basename($filePath);
            $fileExtension = pathinfo($fileName, PATHINFO_EXTENSION);
            
            // Tạo record trong database
            $btsRanking = BtsRanking::create([
                'file_name' => $fileName,
                'file_type' => $fileExtension,
                'status' => 'pending',
                'description' => $description,
                'uploaded_by' => Auth::id(),
            ]);

            // Dispatch job để xử lý file
            ProcessBtsRankingJob::dispatch($btsRanking->id, $filePath, Auth::id());
            
            Notification::make()
                ->title('Upload thành công')
                ->body('File đã được upload và đang được xử lý...')
                ->success()
                ->send();
                
        } catch (\Exception $e) {
            Notification::make()
                ->title('Lỗi upload file')
                ->body('Không thể upload file: ' . $e->getMessage())
                ->danger()
                ->send();
        }
    }

    private function generateExcelContent(array $columns): string
    {
        // Tạo file tạm để tránh lỗi memory
        $tempFile = tempnam(sys_get_temp_dir(), 'bts_ranking_template');
        
        try {
            $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
            $worksheet = $spreadsheet->getActiveSheet();
            
            // Thêm headers
            $col = 'A';
            foreach ($columns as $header) {
                $worksheet->setCellValue($col . '1', $header);
                $col++;
            }
            
            // Tự động giãn column theo nội dung
            foreach (range('A', $col) as $columnID) {
                $worksheet->getColumnDimension($columnID)->setAutoSize(true);
            }
            
            // Lưu vào file tạm
            $writer = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);
            $writer->setPreCalculateFormulas(false);
            $writer->save($tempFile);
            
            // Đọc nội dung file
            $content = file_get_contents($tempFile);
            
            return $content;
        } finally {
            // Xóa file tạm
            if (file_exists($tempFile)) {
                unlink($tempFile);
            }
        }
    }
}
