<?php

namespace App\Jobs;

use App\Models\BtsEnergyAnalysis;
use App\Models\BtsEnergyAnalysisDetail;
use App\Models\Bts;
use App\Services\BtsEnergyAnalysisService;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Storage;
use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Filament\Notifications\Notification;
use App\Helpers\FloatPrecisionHelper;

class ProcessBtsEnergyAnalysisJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected $analysisId;
    protected $filePath;
    protected $fileType;
    protected $chunkSize;
    protected int $userId;


    /**
     * Create a new job instance.
     */
    public function __construct($analysisId, $filePath, $fileType, int $userId, $chunkSize = 1000)
    {
        $this->analysisId = $analysisId;
        $this->filePath = $filePath;
        $this->fileType = $fileType;
        $this->chunkSize = $chunkSize;
        $this->userId = $userId;
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        try {
            ini_set('memory_limit', '-1'); // -1 = vô hạn
            set_time_limit(0); // cho phép chạy không giới hạn thời gian
            ini_set('precision', 20); // Tăng precision để giữ nguyên độ chính xác của số float

            Log::info("Bắt đầu xử lý file BTS Energy Analysis ID: {$this->analysisId}");
            
            $analysis = BtsEnergyAnalysis::find($this->analysisId);
            if (!$analysis) {
                Log::error("Không tìm thấy BtsEnergyAnalysis với ID: {$this->analysisId}");
                return;
            }

            // Cập nhật trạng thái đang xử lý
            $analysis->update(['status' => 'processing']);

            // Đọc file và xử lý theo chunk (đơn giản hóa)
            $result = $this->processFileInChunks();

            // Sau khi đã xử lý xong file, lưu dữ liệu chi tiết vào bảng details
            if ($result['success'] && isset($result['details'])) {
                $this->saveDetailsInChunks($result['details'], $analysis);
            }
            
            if ($result['success']) {
                $analysis->update([
                    'status' => 'completed',
                    'result_file' => $result['result_file'],
                    'processed_count' => $result['processed_count'],
                ]);
                Notification::make()
                    ->title('Xử lý file phân loại trạm theo lưu lượng')
                    ->body('Bạn có thể tải xuống tệp của mình tại đây.')
                    ->success()
                    ->actions([
                        \Filament\Notifications\Actions\Action::make('Tải xuống')
                            ->url(url('/download-result/' . basename($result['result_file'])))
                            ->openUrlInNewTab(),
                    ])
                    ->sendToDatabase(\App\Models\User::find($this->userId));

                Log::info("Xử lý thành công {$result['processed_count']} bản ghi cho Analysis ID: {$this->analysisId}");
            } else {
                $analysis->update([
                    'status' => 'error',
                ]);
                Log::error("Lỗi xử lý Analysis ID {$this->analysisId}: {$result['error']}");
            }
            
        } catch (\Exception $e) {
            Log::error("Lỗi trong ProcessBtsEnergyAnalysisJob: " . $e->getMessage());
            
            if (isset($analysis)) {
                $analysis->update([
                    'status' => 'error',
                ]);
            }
        }
    }

    /**
     * Xử lý file theo chunk để tránh memory overflow
     */
    private function processFileInChunks()
    {
        try {
            $service = new BtsEnergyAnalysisService();
            
            // Tìm file ở đúng vị trí
            $fullFilePath = $service->findActualFilePath($this->filePath);
            $extension = pathinfo($fullFilePath, PATHINFO_EXTENSION);
            
            // Đọc headers trước
            $headers = $this->getFileHeaders($fullFilePath, $extension);
            
            // Tạo file kết quả tạm thời
            $tempResultPath = $service->createTempResultFile($this->fileType, $fullFilePath);
            
            $processedCount = 0;
            $allProcessedData = [];
            
            // Thu thập tất cả dữ liệu trước để tính energy class
            $chunkNumber = 0;
            $service->processFileChunks($fullFilePath, $extension, function($chunkData) use (&$processedCount, &$chunkNumber, &$allProcessedData, $service, $tempResultPath) {
                $chunkNumber++;
                Log::info("Xử lý chunk {$chunkNumber} với " . count($chunkData) . " bản ghi");
                
                // Xử lý chunk này
                $processedChunk = [];
                foreach ($chunkData as $row) {
                    $processedRow = $service->processRow($row, $this->fileType);
                    $processedChunk[] = $processedRow;
                    $allProcessedData[] = $processedRow; // Lưu để tính energy class sau
                }
                
                $processedCount += count($processedChunk);
                
                // Ghi chunk này trực tiếp vào file để tiết kiệm memory
                $service->appendDataToFile($processedChunk, $tempResultPath, $this->fileType);
                
                // Giải phóng memory
                unset($processedChunk);
            }, $this->chunkSize);
            
            // Tính energy class cho tất cả dữ liệu sử dụng service có sẵn
            Log::info("Bắt đầu tính toán energy class cho {$processedCount} bản ghi");
            $allProcessedData = $service->calculateEnergyClass($allProcessedData);
            
            // Insert tất cả dữ liệu vào database với energy class đã tính
            $this->insertAllDataToDatabase($allProcessedData);
            
            // Cập nhật file Excel với energy class đã tính
            $service->updateExcelWithCalculatedData($tempResultPath, $allProcessedData, $this->fileType);
            
            // Tạo file kết quả cuối cùng
            $finalResultPath = $service->createFinalResultFile($tempResultPath);
            
            return [
                'success' => true,
                'result_file' => $finalResultPath,
                'processed_count' => $processedCount
            ];
            
        } catch (\Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Lấy headers của file
     */
    private function getFileHeaders($filePath, $extension)
    {
        if ($extension === 'csv') {
            $handle = fopen($filePath, 'r');
            $headers = fgetcsv($handle);
            fclose($handle);
            return $headers;
        } else {
            $spreadsheet = IOFactory::load($filePath);
            $worksheet = $spreadsheet->getActiveSheet();
            $highestColumn = $worksheet->getHighestColumn();
            
            $headers = [];
            for ($col = 'A'; $col <= $highestColumn; $col++) {
                $headers[] = $worksheet->getCell($col . '1')->getValue();
            }
            return $headers;
        }
    }

    /**
     * Chuẩn bị dữ liệu cho detail table
     */
    private function prepareDetailRow($processedRow, $analysisId)
    {
        $processedRow = (new BtsEnergyAnalysisService())->normalizeNumericValues($processedRow);
        return [
            'bts_energy_analysis_id' => $analysisId,
            'bts_id' => $this->getOrCreateBtsId($processedRow['BTS_id'] ?? ''),
            'month_id' => $processedRow['Month_id'] ?? null,
            'traffic_2g' => $processedRow['Traffic_2G'] ?? null,
            'data_2g' => $processedRow['Data_2G'] ?? null,
            'amr_traf' => $processedRow['AMR_TRAF'] ?? null,
            'data_3g' => $processedRow['Data3G'] ?? null,
            'traffic_4g' => $processedRow['Traffic_4G'] ?? null,
            'p_subscribeduser' => $processedRow['P_SubscribedUser'] ?? null,
            'p_voicetraffic' => $processedRow['P_VoiceTraffic'] ?? null,
            'p_datatraffic' => $processedRow['P_DataTraffic'] ?? null,
            'e_powerconsumption_ac' => $processedRow['E_PowerConsumption_AC'] ?? null,
            'e_powerconsumption_tbvt' => $processedRow['E_PowerConsumption_TBVT'] ?? null,
            'e_powerconsumption_tbtd' => $processedRow['E_PowerConsumption_TBTD'] ?? null,
            'e_powerconsumption_maylanh' => $processedRow['E_PowerConsumption_MAYLANH'] ?? null,
            'distinct_attribute_pin_nlmt' => $processedRow['Distinct_Attribute_PIN_NLMT'] ?? null,
            'ei_user_power' => $processedRow['EI_USER_POWER'] ?? null,
            'ei_voice_power' => $processedRow['EI_VOICE_POWER'] ?? null,
            'ei_data_power' => $processedRow['EI_DATA_POWER'] ?? null,
            'ei_service_power' => $processedRow['EI_SERVICE_POWER'] ?? null,
            'qcut_norm_energy_class' => $processedRow['qcut_norm_ENERGY_CLASS'] ?? null,
            'energy_class' => $processedRow['ENERGY_CLASS'] ?? null,
            'e_contributionratio_tbvt_ac' => $processedRow['E_ContributionRatio_TBVT_AC'] ?? null,
            'e_contributionratio_tbtd_ac' => $processedRow['E_ContributionRatio_TBTD_AC'] ?? null,
            'e_contributionratio_maylanh_ac' => $processedRow['E_ContributionRatio_MAYLANH_AC'] ?? null,
        ];
    }


    /**
     * Lưu details theo chunk (backup method)
     */
    private function saveDetailsInChunks($details, $analysis)
    {
        $detailChunkSize = 1000;
        $chunks = array_chunk($details, $detailChunkSize);
        
        foreach ($chunks as $chunk) {
            foreach ($chunk as &$row) {
                $row['bts_energy_analysis_id'] = $analysis->id;
                if (isset($row['bts_code'])) {
                    $bts = Bts::firstOrCreate(['code' => $row['bts_code']]);
                    $row['bts_id'] = $bts->id;
                    unset($row['bts_code']);
                }
            }
            unset($row);
            BtsEnergyAnalysisDetail::insert($chunk);
            Log::info("Inserted " . count($chunk) . " detail records for BtsEnergyAnalysis");
        }
    }

    
    /**
     * Insert tất cả dữ liệu vào database với energy class đã tính
     */
    private function insertAllDataToDatabase($allProcessedData)
    {
        try {
            Log::info("Bắt đầu insert " . count($allProcessedData) . " bản ghi vào database");
            
            DB::transaction(function () use ($allProcessedData) {
                $service = new BtsEnergyAnalysisService();
                $service->insertAllDetailsToDatabase($allProcessedData, $this->analysisId);
            });
            
            Log::info("Đã hoàn thành insert tất cả dữ liệu vào database");
            
        } catch (\Exception $e) {
            Log::error("Lỗi khi insert dữ liệu vào database: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Lấy hoặc tạo BTS ID từ BTS code
     */
    private function getOrCreateBtsId($btsCode)
    {
        $bts = Bts::firstOrCreate(['code' => $btsCode]);
        return $bts->id;
    }
    
}
