<?php

namespace App\Jobs;

use App\Models\BtsPowerConsumption;
use App\Models\BtsPowerConsumptionDetail;
use App\Models\Bts;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Filament\Notifications\Notification;

class ProcessBtsPowerConsumptionJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected $btsPowerConsumptionId;
    protected $filePath;
    protected $templateType;
    protected int $userId;
    protected array $thresholds;

    public function __construct($btsPowerConsumptionId, $filePath, $templateType, $userId, $thresholds)
    {
        $this->btsPowerConsumptionId = $btsPowerConsumptionId;
        $this->filePath = $filePath;
        $this->templateType = $templateType;
        $this->userId = $userId;
        $this->thresholds = $thresholds;
    }

    public function handle()
    {
        ini_set('memory_limit', '-1'); // -1 = vô hạn
        set_time_limit(0); // cho phép chạy không giới hạn thời gian

        $btsPowerConsumption = BtsPowerConsumption::find($this->btsPowerConsumptionId);
        if (!$btsPowerConsumption) {
            Log::error("BtsPowerConsumption not found: {$this->btsPowerConsumptionId}");
            return;
        }

        try {
            // Cập nhật status thành processing
            $btsPowerConsumption->update(['status' => 'processing']);

            // Đọc file Excel - Filament thường lưu trong private/
            $fullFilePath = storage_path('app/private/' . $this->filePath);
            
            // Kiểm tra file có tồn tại không
            if (!file_exists($fullFilePath)) {
                // Thử đường dẫn không có private/
                $normalFilePath = storage_path('app/' . $this->filePath);
                if (file_exists($normalFilePath)) {
                    $fullFilePath = $normalFilePath;
                } else {
                    throw new \Exception("File không tồn tại: {$fullFilePath} hoặc {$normalFilePath}");
                }
            }
            
            $spreadsheet = IOFactory::load($fullFilePath);
            $worksheet = $spreadsheet->getActiveSheet();
            $data = $worksheet->toArray();

            if (empty($data)) {
                throw new \Exception('File Excel trống hoặc không hợp lệ');
            }

            $header = $data[0];
            $rows = array_slice($data, 1);

            // Tìm index các cột cần thiết
            $radioIndex = array_search('Radio_power_consumption_max', $header);
            $txIndex = array_search('TX_power_consumption_max', $header);
            $coolingIndex = array_search('Cooling_power_consumption_max', $header);
            $btsIdIndex = array_search('BTS_id', $header);

            if ($radioIndex === false || $btsIdIndex === false) {
                throw new \Exception('File thiếu cột bắt buộc: Radio_power_consumption_max hoặc BTS_id');
            }

            // Xử lý theo chunk 1000 records
            $chunkSize = 1000;
            $totalRows = count($rows);
            $processedRows = 0;
            $detailChunkSize = 1000; // Chunk size cho việc lưu detail
            $detailBatch = [];

            DB::transaction(function () use ($rows, $chunkSize, $totalRows, $header, $radioIndex, $txIndex, $coolingIndex, $btsIdIndex, $btsPowerConsumption, $worksheet, &$processedRows, &$detailBatch, $detailChunkSize) {
                for ($i = 0; $i < $totalRows; $i += $chunkSize) {
                    $chunk = array_slice($rows, $i, $chunkSize);
                    
                    foreach ($chunk as $index => $row) {
                        if (empty($row[$btsIdIndex])) {
                            continue; // Bỏ qua dòng trống
                        }

                        $btsId = $row[$btsIdIndex];
                        $radioPower = $this->getNumericValue($row[$radioIndex] ?? 0);
                        $txPower = $this->getNumericValue($row[$txIndex] ?? 0);
                        $coolingPower = $this->getNumericValue($row[$coolingIndex] ?? 0);

                        // Tính BTS_power_consumption_max theo loại template
                        $btsPowerMax = $this->calculateBtsPowerMax(
                            $this->templateType,
                            $radioPower,
                            $txPower,
                            $coolingPower
                        );

                        // Tính power_consumption_class
                        $powerClass = $this->calculatePowerClass(
                            $this->templateType,
                            $btsPowerMax
                        );

                        // Cập nhật vào file Excel
                        $this->updateExcelRow($worksheet, $i + $index + 2, $header, $btsPowerMax, $powerClass);
                        
                        // Chuẩn bị dữ liệu cho detail table
                        $detailRow = [
                            'bts_power_consumption_id' => $btsPowerConsumption->id,
                            'bts_id' => $this->getOrCreateBtsId($btsId),
                            'radio_power_consumption_max' => $radioPower,
                            'tx_power_consumption_max' => $txPower,
                            'cooling_power_consumption_max' => $coolingPower,
                            'bts_type' => $row[array_search('BTS_type', $header)] ?? null,
                            'radio_config_3g' => $row[array_search('Radio_config_3G', $header)] ?? null,
                            'radio_config_4g' => $row[array_search('Radio_config_4G', $header)] ?? null,
                            'radio_config_mimo_4g' => $row[array_search('Radio_config_MIMO_4G', $header)] ?? null,
                            'radio_type' => $row[array_search('Radio_type', $header)] ?? null,
                            'radio_model_name' => $row[array_search('Radio_model_name', $header)] ?? null,
                            'radio_family_id' => $this->getNumericValue($row[array_search('Radio_family_id', $header)] ?? 0),
                            'tx_type' => $row[array_search('TX_type', $header)] ?? null,
                            'tx_brand_name' => $row[array_search('TX_brand_name', $header)] ?? null,
                            'tx_model_name' => $row[array_search('TX_model_name', $header)] ?? null,
                            'tx_family_id' => $this->getNumericValue($row[array_search('TX_family_id', $header)] ?? 0),
                            'cooling_brand_name' => $row[array_search('Cooling_brand_name', $header)] ?? null,
                            'cooling_family_id' => $this->getNumericValue($row[array_search('Cooling_family_id', $header)] ?? 0),
                            'radio_family_type' => $row[array_search('Radio_family_type', $header)] ?? null,
                            'tx_family_type' => $row[array_search('TX_family_type', $header)] ?? null,
                            'cooling_family_type' => $row[array_search('Cooling_family_type', $header)] ?? null,
                            'power_consumption_class' => $powerClass,
                            'bts_power_consumption_max' => $btsPowerMax,

                        ];
                        
                        $detailBatch[] = $detailRow;
                        
                        // Lưu detail theo chunk
                        if (count($detailBatch) >= $detailChunkSize) {
                            BtsPowerConsumptionDetail::insert($detailBatch);
                            Log::info("Inserted {$detailChunkSize} detail records for BtsPowerConsumption {$this->btsPowerConsumptionId}");
                            $detailBatch = [];
                        }
                        
                        $processedRows++;
                    }

                    // Log tiến độ xử lý
                    Log::info("Processed chunk: " . ($i + $chunkSize) . "/{$totalRows} rows for BtsPowerConsumption {$this->btsPowerConsumptionId}");
                    
                    // Giải phóng memory sau mỗi chunk
                    if (function_exists('gc_collect_cycles')) {
                        gc_collect_cycles();
                    }
                }
                
                // Lưu detail batch cuối cùng
                if (!empty($detailBatch)) {
                    BtsPowerConsumptionDetail::insert($detailBatch);
                    Log::info("Inserted " . count($detailBatch) . " final detail records for BtsPowerConsumption {$this->btsPowerConsumptionId}");
                }
            });

            // Lưu file đã cập nhật
            $writer = new Xlsx($spreadsheet);
            $writer->save($fullFilePath);

            // Cập nhật status thành completed
            $btsPowerConsumption->update([
                'status' => 'completed',
                'processed_at' => now(),
                'processed_count' => $processedRows
            ]);

            Notification::make()
                ->title('Xử lý file phân loại trạm theo phần tử hoàn tất')
                ->body('Bạn có thể tải xuống tệp của mình tại đây.')
                ->success()
                ->actions([
                    \Filament\Notifications\Actions\Action::make('Tải xuống')
                    ->url(url('/bts-power/download/' . $this->btsPowerConsumptionId))
                    ->openUrlInNewTab(),
                ])
                ->sendToDatabase(\App\Models\User::find($this->userId));

            Log::info("Processed BtsPowerConsumption successfully: {$this->btsPowerConsumptionId}");

        } catch (\Exception $e) {
            Log::error("Error processing BtsPowerConsumption {$this->btsPowerConsumptionId}: " . $e->getMessage());
            
            $btsPowerConsumption->update([
                'status' => 'failed',
                'description' => 'Lỗi xử lý: ' . $e->getMessage()
            ]);
        }
    }

    private function getNumericValue($value)
    {
        if (is_numeric($value)) {
            return (float) $value;
        }

        // Xử lý trường hợp có ký tự không phải số
        if (preg_match('/[\d\.]+/', $value, $matches)) {
            return (float) $matches[0];
        }

        return 0;
    }

    private function calculateBtsPowerMax($templateType, $radioPower, $txPower, $coolingPower)
    {
        switch ($templateType) {
            case 'full':
                return $radioPower + $txPower + $coolingPower;
            case 'no_cooling':
                return $radioPower + $txPower;
            case 'no_tx':
                return $radioPower + $coolingPower;
            default:
                return $radioPower;
        }
    }

    private function calculatePowerClass($templateType, $btsPowerMax)
    {
        $defaultRules = [
            'full' => [
                ['class' => 'low_medium', 'min' => null,    'max' => 9400],
                ['class' => 'medium_high', 'min' => 9400,   'max' => 20000],
                ['class' => 'high',        'min' => 20000,  'max' => 50000],
                ['class' => 'extreme',     'min' => 50000,  'max' => null],
            ],
            'no_cooling' => [
                ['class' => 'low_medium', 'min' => null,    'max' => 6500],
                ['class' => 'medium_high', 'min' => 6500,   'max' => 15000],
                ['class' => 'high',        'min' => 15000,  'max' => 35000],
                ['class' => 'extreme',     'min' => 35000,  'max' => null],
            ],
            'no_tx' => [
                ['class' => 'low_medium', 'min' => null,    'max' => 5500],
                ['class' => 'medium_high', 'min' => 5500,   'max' => 12000],
                ['class' => 'high',        'min' => 12000,  'max' => 25000],
                ['class' => 'extreme',     'min' => 25000,  'max' => null],
            ],
        ];
        $thresholds = $this->thresholds;
        $rules = [];
        if (isset($defaultRules[$templateType])) {
            foreach ($defaultRules[$templateType] as $rule) {
                $class = $rule['class'];
                $min = $thresholds[$class]['min'] ?? $rule['min'];
                $max = $thresholds[$class]['max'] ?? $rule['max'];
                $rules[] = [ 'class' => $class, 'min' => $min, 'max' => $max ];
            }
        } else {
            return 'N/A';
        }
        foreach ($rules as $i => $rule) {
            $minOk = $rule['min'] === null || $btsPowerMax > $rule['min'];
            $maxOk = $rule['max'] === null || $btsPowerMax <= $rule['max'];
            if ($minOk && $maxOk) return $rule['class'];
        }
        return 'N/A';
    }

    private function updateExcelRow($worksheet, $rowIndex, $header, $btsPowerMax, $powerClass)
    {
        // Đếm số cột có dữ liệu thực sự (không trống)
        $totalColumns = 0;
        foreach ($header as $column) {
            if (!empty(trim($column))) {
                $totalColumns++;
            }
        }
        
        // Thêm header cho 2 cột mới (chỉ thêm 1 lần cho dòng đầu)
        if ($rowIndex == 2) { // Dòng đầu tiên của data (sau header)
            $btsPowerMaxColumnLetter = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($totalColumns + 1);
            $powerClassColumnLetter = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($totalColumns + 2);
            
            $worksheet->setCellValue($btsPowerMaxColumnLetter . '1', 'BTS_power_consumption_max');
            $worksheet->setCellValue($powerClassColumnLetter . '1', 'power_consumption_class');
        }

        // Cập nhật giá trị cho dòng hiện tại - thêm trực tiếp vào cột tiếp theo
        $btsPowerMaxColumnLetter = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($totalColumns + 1);
        $powerClassColumnLetter = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($totalColumns + 2);
        
        $worksheet->setCellValue($btsPowerMaxColumnLetter . $rowIndex, $btsPowerMax);
        $worksheet->setCellValue($powerClassColumnLetter . $rowIndex, $powerClass);
    }

    /**
     * Lấy hoặc tạo BTS ID từ BTS code
     */
    private function getOrCreateBtsId($btsCode)
    {
        $bts = Bts::firstOrCreate(['code' => $btsCode]);
        return $bts->id;
    }
}
