<?php

namespace App\Jobs;

use App\Models\BtsEnergyAnalysis;
use App\Models\BtsEnergyAnalysisDetail;
use App\Models\Bts;
use App\Services\BtsEnergyAnalysisService;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use Filament\Notifications\Notification;

class ProcessNewBtsWithReferenceJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected int $analysisId;
    protected string $filePath;
    protected string $fileType;
    protected int $referenceAnalysisId;
    protected int $userId;
    protected int $chunkSize;

    public function __construct(
        int $analysisId,
        string $filePath,
        string $fileType,
        int $referenceAnalysisId,
        int $userId,
        int $chunkSize = 1000
    ) {
        $this->analysisId = $analysisId;
        $this->filePath = $filePath;
        $this->fileType = $fileType;
        $this->referenceAnalysisId = $referenceAnalysisId;
        $this->userId = $userId;
        $this->chunkSize = $chunkSize;
    }

    public function handle(): void
    {
        try {
            ini_set('memory_limit', '-1');
            set_time_limit(0);

            Log::info("Bắt đầu xử lý BTS mới theo tham chiếu ID: {$this->referenceAnalysisId} cho Analysis ID: {$this->analysisId}");

            $analysis = BtsEnergyAnalysis::find($this->analysisId);
            if (!$analysis) {
                Log::error("Không tìm thấy BtsEnergyAnalysis với ID: {$this->analysisId}");
                return;
            }

            $analysis->update(['status' => 'processing']);

            $result = $this->processUsingReference();

            if ($result['success']) {
                $analysis->update([
                    'status' => 'completed',
                    'result_file' => $result['result_file'],
                    'processed_count' => $result['processed_count'],
                ]);

                Notification::make()
                    ->title('Xử lý file mới theo file trước đó')
                    ->body('Bạn có thể tải xuống tệp của mình tại đây.')
                    ->success()
                    ->actions([
                        \Filament\Notifications\Actions\Action::make('Tải xuống')
                            ->url(url('/download-result/' . basename($result['result_file'])))
                            ->openUrlInNewTab(),
                    ])
                    ->sendToDatabase(\App\Models\User::find($this->userId));
            } else {
                $analysis->update(['status' => 'error']);
                Log::error("Lỗi xử lý Analysis ID {$this->analysisId}: {$result['error']}");
            }
        } catch (\Exception $e) {
            Log::error("Lỗi ProcessNewBtsWithReferenceJob: " . $e->getMessage());
            if (isset($analysis)) {
                $analysis->update(['status' => 'error']);
            }
        }
    }

    private function processUsingReference(): array
    {
        try {
            $service = new BtsEnergyAnalysisService();

            $fullFilePath = $service->findActualFilePath($this->filePath);
            $extension = pathinfo($fullFilePath, PATHINFO_EXTENSION);

            // Tạo file kết quả tạm thời (giữ format như job gốc)
            $tempResultPath = $service->createTempResultFile($this->fileType, $fullFilePath);

            $processedCount = 0;
            $allProcessedData = [];

            // Đọc theo chunk, xử lý EI_* và các metric cơ bản
            $service->processFileChunks($fullFilePath, $extension, function ($chunkData) use (&$processedCount, &$allProcessedData, $service, $tempResultPath) {
                $processedChunk = [];
                foreach ($chunkData as $row) {
                    $processedRow = $service->processRow($row, $this->fileType);
                    $processedChunk[] = $processedRow;
                    $allProcessedData[] = $processedRow;
                }
                $processedCount += count($processedChunk);
                $service->appendDataToFile($processedChunk, $tempResultPath, $this->fileType);
            }, $this->chunkSize);

            // Lấy EI_SERVICE_POWER tham chiếu và kết hợp với giá trị của file mới
            $referenceValues = $this->getReferenceEiServicePowers($this->referenceAnalysisId);
            $newValues = collect($allProcessedData)
                ->pluck('EI_SERVICE_POWER')
                ->map(function ($v) { return (float)($v ?? 0); })
                ->filter(function ($v) { return is_finite($v); })
                ->values()
                ->toArray();

            $combinedValues = array_values(array_filter(array_merge($referenceValues, $newValues), function ($v) {
                return is_finite($v);
            }));

            if (empty($combinedValues)) {
                throw new \Exception('Không có dữ liệu EI_SERVICE_POWER hợp lệ từ tham chiếu và file mới.');
            }
            // Log::info('=== ENERGY CLASS DEBUG ===', [
            //     'combinedValues_count' => count($combinedValues),
            //     'combinedValues_sample' => array_slice($combinedValues, 0, 10), // log 10 giá trị đầu tiên
            //     'minVal' => min($combinedValues),
            //     'maxVal' => max($combinedValues),
            // ]);

            // Tính min-max và quantiles trên tập kết hợp (tham chiếu + file mới)
            $minVal = min($combinedValues);
            $maxVal = max($combinedValues);
            $quantiles = $service->calculateQuantiles($combinedValues);

            // Áp dụng chuẩn hóa min-max và phân loại theo quantiles kết hợp
            foreach ($allProcessedData as &$row) {
                $value = (float) ($row['EI_SERVICE_POWER'] ?? 0);
                if ($maxVal > $minVal) {
                    $row['qcut_norm_ENERGY_CLASS'] = (string)(100 * ($value - $minVal) / ($maxVal - $minVal));
                } else {
                    $row['qcut_norm_ENERGY_CLASS'] = '50';
                }
                $row['ENERGY_CLASS'] = $service->getEnergyClassByQuantile($value, $quantiles);
            }
            unset($row);

            // Insert DB và cập nhật file kết quả
            $service->insertAllDetailsToDatabase($allProcessedData, $this->analysisId);
            $service->updateExcelWithCalculatedData($tempResultPath, $allProcessedData, $this->fileType);
            $finalResultPath = $service->createFinalResultFile($tempResultPath);

            return [
                'success' => true,
                'result_file' => $finalResultPath,
                'processed_count' => $processedCount,
            ];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    private function getReferenceEiServicePowers(int $refId): array
    {
        return BtsEnergyAnalysisDetail::where('bts_energy_analysis_id', $refId)
            ->whereNotNull('ei_service_power')
            ->pluck('ei_service_power')
            ->map(function ($v) {
                return (float)$v;
            })
            ->filter(function ($v) {
                return is_finite($v);
            })
            ->values()
            ->toArray();
    }
}


