<?php

namespace App\Services;

use Illuminate\Support\Facades\Storage;
use Maatwebsite\Excel\Facades\Excel;
use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use Illuminate\Support\Facades\Log;
use App\Helpers\FloatPrecisionHelper;

class BtsEnergyAnalysisService
{

    public function processRow($row, $fileType)
    {
        // Chuẩn hóa tên cột
        $normalizedRow = $this->normalizeColumnNames($row);
        
        // Tính toán các metrics
        $processedRow = $this->calculateMetrics($normalizedRow, $fileType);
        
        return $processedRow;
    }
    
    private function normalizeColumnNames($row)
    {
        $normalized = [];
        
        foreach ($row as $key => $value) {
            // Chuẩn hóa tên cột
            $normalizedKey = str_replace([' ', '-'], '_', strtolower($key));
            $normalized[$normalizedKey] = $value;
        }
        
        return $normalized;
    }
    
    private function calculateMetrics($row, $fileType)
    {
        // Lấy các giá trị cơ bản
        $btsId = $row['bts_id'] ?? '';
        $monthId = $row['month_id'] ?? null;
        $pSubscribedUser = $row['p_subscribed_user'] ?? $row['p_subscribeduser'] ?? 0;
        $pVoiceTraffic = $row['p_voice_traffic'] ?? $row['p_voicetraffic'] ?? 0;
        $pDataTraffic = $row['p_data_traffic'] ?? $row['p_datatraffic'] ?? 0;
        $ePowerConsumptionAc = $row['e_power_consumption_ac'] ?? $row['e_powerconsumption_ac'] ?? 0;
        $ePowerConsumptionTbvt = $row['e_power_consumption_tbvt'] ?? $row['e_powerconsumption_tbvt'] ?? 0;
        $ePowerConsumptionTbtd = $row['e_power_consumption_tbtd'] ?? $row['e_powerconsumption_tbtd'] ?? 0;
        $ePowerConsumptionMaylanh = $row['e_power_consumption_maylanh'] ?? $row['e_powerconsumption_maylanh'] ?? 0;
        
        // Nếu có month_id, tính P_VoiceTraffic và P_DataTraffic
        if ($monthId) {
            $traffic2g = $row['traffic_2g'] ?? 0;
            $data2g = $row['data_2g'] ?? 0;
            $amrTraf = $row['amr_traf'] ?? 0;
            $data3g = $row['data3g'] ?? 0;
            $traffic4g = $row['traffic_4g'] ?? 0;
            
            $pVoiceTraffic = $traffic2g + $amrTraf;
            $pDataTraffic = $data2g + $data3g + $traffic4g;
        }
        
        // Tính EI metrics với độ chính xác cao - sử dụng số float gốc
        $eiUserPower = ($ePowerConsumptionAc > 0 && is_numeric($pSubscribedUser) && is_numeric($ePowerConsumptionAc)) 
            ? (string)($pSubscribedUser !== '' && $ePowerConsumptionAc !== '' ? sqrt($pSubscribedUser / $ePowerConsumptionAc) : 0)
            : '0';
        $eiVoicePower = ($ePowerConsumptionAc > 0 && is_numeric($pVoiceTraffic) && is_numeric($ePowerConsumptionAc))
            ? (string)($pVoiceTraffic !== '' && $ePowerConsumptionAc !== '' ? sqrt($pVoiceTraffic / $ePowerConsumptionAc) : 0)
            : '0';
        $eiDataPower = ($ePowerConsumptionAc > 0 && is_numeric($pDataTraffic) && is_numeric($ePowerConsumptionAc))
            ? (string)($pDataTraffic !== '' && $ePowerConsumptionAc !== '' ? sqrt($pDataTraffic / $ePowerConsumptionAc / 10000) : 0)
            : '0';
        $eiServicePower = ($ePowerConsumptionAc > 0 && is_numeric($pVoiceTraffic) && is_numeric($pDataTraffic) && is_numeric($ePowerConsumptionAc))
            ? (string)($pVoiceTraffic !== '' && $pDataTraffic !== '' && $ePowerConsumptionAc !== '' ? 
                sqrt($pDataTraffic / $ePowerConsumptionAc / 10000) + sqrt($pVoiceTraffic * 5490 / $ePowerConsumptionAc / 1000) : 0)
            : '0';
        
        // Tính E_ContributionRatio với độ chính xác cao - sử dụng số float gốc
        $eContributionRatioTbvtAc = $ePowerConsumptionAc > 0 ? 
            (string)(100 * $ePowerConsumptionTbvt / ($ePowerConsumptionAc * 1000 / 720)) : '0';
        $eContributionRatioTbtdAc = $ePowerConsumptionAc > 0 ? 
            (string)(100 * $ePowerConsumptionTbtd / ($ePowerConsumptionAc * 1000 / 720)) : '0';
        $eContributionRatioMaylanhAc = $ePowerConsumptionAc > 0 ? 
            (string)(100 * $ePowerConsumptionMaylanh / ($ePowerConsumptionAc * 1000 / 720)) : '0';
        
        // qcut_norm_ENERGY_CLASS sẽ được tính sau khi có tất cả dữ liệu
        $qcutNormEnergyClass = 0; // Tạm thời, sẽ tính sau
        
        $result = [
            'STT' => $row['stt'] ?? $row['STT'] ?? 0, // Lấy STT từ file input
            'BTS_id' => $btsId,
            'Month_id' => $monthId,
            'P_SubscribedUser' => $pSubscribedUser,
            'P_VoiceTraffic' => $pVoiceTraffic,
            'P_DataTraffic' => $pDataTraffic,
            'E_PowerConsumption_AC' => $ePowerConsumptionAc,
            'E_PowerConsumption_TBVT' => $ePowerConsumptionTbvt,
            'E_PowerConsumption_TBTD' => $ePowerConsumptionTbtd,
            'E_PowerConsumption_MAYLANH' => $ePowerConsumptionMaylanh,
            'EI_USER_POWER' => $eiUserPower,
            'EI_VOICE_POWER' => $eiVoicePower,
            'EI_DATA_POWER' => $eiDataPower,
            'EI_SERVICE_POWER' => $eiServicePower,
            'qcut_norm_ENERGY_CLASS' => $qcutNormEnergyClass,
            'E_ContributionRatio_TBVT_AC' => $eContributionRatioTbvtAc,
            'E_ContributionRatio_TBTD_AC' => $eContributionRatioTbtdAc,
            'E_ContributionRatio_MAYLANH_AC' => $eContributionRatioMaylanhAc,
            'Distinct_Attribute_PIN_NLMT' => $row['distinct_attribute_pin_nlmt'] ?? '',
            'ENERGY_CLASS' => $row['energy_class'] ?? '', // Giữ nguyên từ input
        ];
        
        // Thêm các trường traffic cho file có month_id
        if ($monthId) {
            $result['Traffic_2G'] = $traffic2g;
            $result['Data_2G'] = $data2g;
            $result['AMR_TRAF'] = $amrTraf;
            $result['Data3G'] = $data3g;
            $result['Traffic_4G'] = $traffic4g;
        }
        
        return $result;
    }
    
    public function calculateEnergyClass($data)
    {
        // Lấy tất cả giá trị EI_SERVICE_POWER
        $eiServicePowers = array_column($data, 'EI_SERVICE_POWER');
        
        // Tìm min và max của EI_SERVICE_POWER
        $minEiServicePower = min($eiServicePowers);
        $maxEiServicePower = max($eiServicePowers);
        
        // Tính các phân vị (quantiles) cho 6 mức năng lượng
        $quantiles = $this->calculateQuantiles($eiServicePowers);
        
        // Tính qcut_norm_ENERGY_CLASS và ENERGY_CLASS cho từng record
        foreach ($data as &$row) {
            $eiServicePower = $row['EI_SERVICE_POWER'];
            
            // Cách 1: Chuẩn hóa EI_SERVICE_POWER về dải [0,100] với độ chính xác cao
            // EI_SERVICE_POWER_normalized = 100*(EI_SERVICE_POWER-min)/(max-min)
            if ($maxEiServicePower > $minEiServicePower) {
                $normalizedValue = (string)(100 * ($eiServicePower - $minEiServicePower) / ($maxEiServicePower - $minEiServicePower));
            } else {
                $normalizedValue = '50'; // Nếu min = max, gán giá trị trung bình
            }
            $row['qcut_norm_ENERGY_CLASS'] = $normalizedValue;
            
            // Cách 2: Tính ENERGY_CLASS dựa trên phân vị (quantile)
            $row['ENERGY_CLASS'] = $this->getEnergyClassByQuantile($eiServicePower, $quantiles);
        }
        
        return $data;
    }
    
    private function calculateQuantiles($values)
    {
        // Sort ascending: EI_SERVICE_POWER cao hơn = hiệu quả tốt hơn
        sort($values);
        $n = count($values);
        
        if ($n === 0) return [];
        
        // Tính 5 điểm phân vị tương ứng 6 mức năng lượng
        $quantiles = [];
        $percentiles = [10, 28, 55, 80, 97]; // 5 điểm phân vị
        
        foreach ($percentiles as $p) {
            // Tính vị trí trong mảng (0-based)
            $index = ($p / 100) * ($n - 1);
            $lower = floor($index);
            $upper = ceil($index);
            
            if ($lower === $upper) {
                $quantiles[$p] = (string)$values[$lower];
            } else {
                // Linear interpolation với độ chính xác cao
                $weight = $index - $lower;
                $quantiles[$p] = (string)($values[$lower] * (1 - $weight) + $values[$upper] * $weight);
            }
        }
        
        return $quantiles;
    }
    
    private function calculatePercentile($value, $values)
    {
        // Sort ascending: EI_SERVICE_POWER cao hơn = hiệu quả tốt hơn
        sort($values);
        $n = count($values);
        
        if ($n === 0) return 0;
        
        // Tìm vị trí của value trong mảng đã sort
        $positions = [];
        for ($i = 0; $i < $n; $i++) {
            if ($values[$i] == $value) {
                $positions[] = $i + 1; // Rank bắt đầu từ 1
            }
        }
        
        if (empty($positions)) {
            // Nếu không tìm thấy, tìm vị trí gần nhất
            $count = 0;
            foreach ($values as $v) {
                if ($v <= $value) {
                    $count++;
                }
            }
            return (string)(($count / $n) * 100);
        }
        
        // Tính rank trung bình cho các giá trị trùng nhau
        $averageRank = array_sum($positions) / count($positions);
        
        return (string)(($averageRank / $n) * 100);
    }
    
    private function getEnergyClassByQuantile($value, $quantiles)
    {
        // Phân loại theo 6 mức năng lượng dựa trên phân vị
        if ($value > $quantiles[97]) {
            return 'A'; // >97%
        } elseif ($value >= $quantiles[80]) {
            return 'B'; // 80%-97%
        } elseif ($value >= $quantiles[55]) {
            return 'C'; // 55%-80%
        } elseif ($value >= $quantiles[28]) {
            return 'D'; // 28%-55%
        } elseif ($value >= $quantiles[10]) {
            return 'E'; // 10%-28%
        } else {
            return 'F'; // <10%
        }
    }

    
}
