<?php

namespace App\Services;

use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use App\Models\Bts;
use App\Models\BtsPowerConsumptionDetail;

class BtsPowerConsumptionService
{
    public function findActualFilePath(string $filePath): string
    {
        $private = storage_path('app/private/' . $filePath);
        if (file_exists($private)) {
            return $private;
        }
        $normal = storage_path('app/' . $filePath);
        if (file_exists($normal)) {
            return $normal;
        }
        throw new \Exception("File không tồn tại: {$private} hoặc {$normal}");
    }

    public function getNumericValue($value): float
    {
        if (is_numeric($value)) {
            return (float) $value;
        }
        if (is_string($value) && preg_match('/[\d\.]+/', $value, $m)) {
            return (float) $m[0];
        }
        return 0.0;
    }

    public function calculateBtsPowerMax(string $templateType, float $radioPower, float $txPower, float $coolingPower): float
    {
        switch ($templateType) {
            case 'full':
                return $radioPower + $txPower + $coolingPower;
            case 'no_cooling':
                return $radioPower + $txPower;
            case 'no_tx':
                return $radioPower + $coolingPower;
            default:
                return $radioPower;
        }
    }

    public function calculatePowerClass(string $templateType, float $btsPowerMax, array $thresholds): string
    {
        $defaultRules = [
            'full' => [
                ['class' => 'low_medium', 'min' => null,   'max' => 9400],
                ['class' => 'medium_high','min' => 9400,   'max' => 20000],
                ['class' => 'high',       'min' => 20000,  'max' => 50000],
                ['class' => 'extreme',    'min' => 50000,  'max' => null],
            ],
            'no_cooling' => [
                ['class' => 'low_medium', 'min' => null,   'max' => 6500],
                ['class' => 'medium_high','min' => 6500,   'max' => 15000],
                ['class' => 'high',       'min' => 15000,  'max' => 35000],
                ['class' => 'extreme',    'min' => 35000,  'max' => null],
            ],
            'no_tx' => [
                ['class' => 'low_medium', 'min' => null,   'max' => 5500],
                ['class' => 'medium_high','min' => 5500,   'max' => 12000],
                ['class' => 'high',       'min' => 12000,  'max' => 25000],
                ['class' => 'extreme',    'min' => 25000,  'max' => null],
            ],
        ];

        $rules = [];
        if (isset($defaultRules[$templateType])) {
            foreach ($defaultRules[$templateType] as $rule) {
                $class = $rule['class'];
                $min = $thresholds[$class]['min'] ?? $rule['min'];
                $max = $thresholds[$class]['max'] ?? $rule['max'];
                $rules[] = ['class' => $class, 'min' => $min, 'max' => $max];
            }
        } else {
            return 'N/A';
        }
        foreach ($rules as $rule) {
            $minOk = $rule['min'] === null || $btsPowerMax > (float)$rule['min'];
            $maxOk = $rule['max'] === null || $btsPowerMax <= (float)$rule['max'];
            if ($minOk && $maxOk) {
                return $rule['class'];
            }
        }
        return 'N/A';
    }

    public function getOrCreateBtsId(string $btsCode): int
    {
        $bts = Bts::firstOrCreate(['code' => $btsCode]);
        return $bts->id;
    }

    public function insertDetailsInChunks(array $rows, int $chunkSize = 1000): void
    {
        $chunks = array_chunk($rows, $chunkSize);
        foreach ($chunks as $chunk) {
            BtsPowerConsumptionDetail::insert($chunk);
        }
    }
}


